local Foundation = script:FindFirstAncestor("Foundation")
local Packages = Foundation.Parent

local Dash = require(Packages.Dash)
local React = require(Packages.React)
local ReactRoblox = require(Packages.ReactRoblox)

local CloseAffordance = require(Foundation.Components.CloseAffordance)
local Constants = require(Foundation.Constants)
local Image = require(Foundation.Components.Image)
local View = require(Foundation.Components.View)

local useOverlay = require(Foundation.Providers.Overlay.useOverlay)

local DialogSize = require(Foundation.Enums.DialogSize)
local OnCloseCallbackReason = require(Foundation.Enums.OnCloseCallbackReason)
local StateLayerAffordance = require(Foundation.Enums.StateLayerAffordance)

local useScaledValue = require(Foundation.Utility.useScaledValue)
local withCommonProps = require(Foundation.Utility.withCommonProps)
local withDefaults = require(Foundation.Utility.withDefaults)

local DialogTypes = require(script.Parent.Types)
local Types = require(Foundation.Components.Types)
local useDialogVariants = require(script.Parent.useDialogVariants).useDialogVariants
local DialogProvider = require(script.Parent.DialogProvider)
local useDialogResponsiveSize = require(script.Parent.useDialogResponsiveSize)
local useDialogSize = require(script.Parent.useDialogSize)

local useElevation = require(Foundation.Providers.Elevation.useElevation)
local OwnerScope = require(Foundation.Providers.Elevation.ElevationProvider).ElevationOwnerScope
local ElevationLayer = require(Foundation.Enums.ElevationLayer)
type ElevationLayer = ElevationLayer.ElevationLayer

type DialogSize = DialogSize.DialogSize
type OnCloseCallbackReason = OnCloseCallbackReason.OnCloseCallbackReason

export type DialogProps = {
	onClose: ((reason: OnCloseCallbackReason?) -> ())?,
	size: DialogSize?,
	disablePortal: boolean?,
	hasBackdrop: boolean?,
	children: React.ReactNode,
	testId: string?,
} & Types.NativeCallbackProps

type DialogInternalProps = {
	forwardRef: React.Ref<GuiObject>?,
} & DialogProps

local defaultProps = {
	size = DialogSize.Medium,
	disablePortal = true,
	hasBackdrop = false,
	testId = "--foundation-dialog",
}

local SHADOW_IMAGE = Constants.SHADOW_IMAGE
local SHADOW_SIZE = Constants.SHADOW_SIZE

local function Dialog(dialogProps: DialogInternalProps)
	local props = Dash.assign({}, dialogProps, { LayoutOrder = 1 })
	local variants = useDialogVariants()
	local maxWidth = useScaledValue(variants.inner.maxWidth)
	local dialogBodyRef = React.useRef(nil)
	local dialogSizeBinding = useDialogSize(dialogBodyRef)
	local overlay = useOverlay()
	local elevation = useElevation(ElevationLayer.Dialog, { stackAboveOwner = false })

	useDialogResponsiveSize(props.size)

	local content = React.createElement(View, {
		ZIndex = elevation.zIndex,
		tag = "size-full",
		testId = `{props.testId}--container`,
	}, {
		Backdrop = if props.hasBackdrop
			then React.createElement(View, {
				tag = "position-center-center anchor-center-center",
				Size = UDim2.fromScale(2, 2),
				stateLayer = {
					affordance = StateLayerAffordance.None,
				},
				onActivated = function()
					if props.onClose then
						props.onClose(OnCloseCallbackReason.BackdropClick)
					end
				end,
				backgroundStyle = variants.backdrop.backgroundStyle,
				ZIndex = 2,
				testId = `{props.testId}--backdrop`,
			})
			else nil,
		DialogShadowWrapper = React.createElement(View, {
			tag = variants.container.tag,
			ZIndex = 2,
		}, {
			Shadow = React.createElement(Image, {
				Image = SHADOW_IMAGE,
				Size = dialogSizeBinding:map(function(size: Vector2): UDim2
					return UDim2.fromOffset(size.X + SHADOW_SIZE * 2, size.Y + SHADOW_SIZE * 2)
				end),
				slice = {
					center = Rect.new(SHADOW_SIZE, SHADOW_SIZE, SHADOW_SIZE, SHADOW_SIZE),
					scale = 2,
				},
				imageStyle = variants.shadow.imageStyle,
				testId = `{props.testId}--shadow`,
			}),
		}),
	}, {
		Dialog = React.createElement(View, {
			tag = variants.container.tag,
			ZIndex = 3,
		}, {
			DialogFlexStart = React.createElement(View, {
				tag = "fill",
				LayoutOrder = 0,
			}),
			DialogInner = React.createElement(
				View,
				withCommonProps(props, {
					tag = variants.inner.tag,
					ref = props.forwardRef,
					sizeConstraint = {
						MaxSize = Vector2.new(maxWidth, math.huge),
					},
					stateLayer = {
						affordance = StateLayerAffordance.None,
					},
					selection = DialogTypes.nonSelectable,
					selectionGroup = DialogTypes.isolatedSelectionGroup,
					-- Needed to sink the onActivated event to the backdrop
					onActivated = Dash.noop,
				}),
				{
					CloseAffordance = if props.onClose
						then React.createElement(CloseAffordance, {
							onActivated = props.onClose,
							Position = UDim2.new(
								1,
								-variants.closeAffordance.offset,
								0,
								variants.closeAffordance.offset
							),
							AnchorPoint = Vector2.new(1, 0),
							ZIndex = 2,
							testId = `{props.testId}--close-affordance`,
						})
						else nil,
					DialogBody = React.createElement(View, {
						tag = variants.body.tag,
						ref = dialogBodyRef,
						testId = `{props.testId}--body`,
					}, React.createElement(OwnerScope, { owner = elevation }, props.children)),
				}
			),
			DialogFlexEnd = React.createElement(View, {
				tag = "fill",
				LayoutOrder = 2,
			}),
		}),
	})

	if props.disablePortal or overlay == nil then
		return content
	end

	return ReactRoblox.createPortal(content, overlay)
end

local function DialogContainer(dialogContainerProps: DialogProps, ref: React.Ref<GuiObject>?)
	local props = withDefaults(dialogContainerProps, defaultProps)

	return React.createElement(DialogProvider, {
		size = props.size,
		responsiveSize = props.size,
		testId = props.testId,
	}, {
		Dialog = React.createElement(
			Dialog,
			Dash.assign({}, props, {
				forwardRef = ref,
			})
		),
	})
end

return React.memo(React.forwardRef(DialogContainer))
