--!nonstrict
local Root = script
local CorePackages = game:GetService("CorePackages")
local PurchasePromptDeps = require(CorePackages.Workspace.Packages.PurchasePromptDeps)
local CoreGui = game:GetService("CoreGui")
local RunService = game:GetService("RunService")

local RobloxGui = CoreGui:WaitForChild("RobloxGui")

local Roact = PurchasePromptDeps.Roact
local Rodux = PurchasePromptDeps.Rodux

local PurchasePromptApp = require(script.Components.PurchasePromptApp)
local Reducer = require(Root.Reducers.Reducer)
local ABTest = require(Root.Services.ABTest)
local Network = require(Root.Services.Network)
local Analytics = require(Root.Services.Analytics)
local PlatformInterface = require(Root.Services.PlatformInterface)
local ExternalSettings = require(Root.Services.ExternalSettings)
local PublicBindables = require(Root.Services.PublicBindables)
local Thunk = require(Root.Thunk)
local initiateAvatarCreationFeePurchaseThunk = require(Root.Thunks.initiateAvatarCreationFeePurchase)
local FFlagHideAvatarIECPromptOnUpsellSuccess = require(RobloxGui.Modules.PublishAssetPrompt.FFlagHideAvatarIECPromptOnUpsellSuccess)

local WindowState = require(Root.Enums.WindowState)
local PromptState = require(Root.Enums.PromptState)
local PublishAssetAnalytics = require(Root.Utils.PublishAssetAnalytics)

local handle
local store

local windowStateChangedBindable = Instance.new("BindableEvent")
local promptStateSetToNoneBindable = Instance.new("BindableEvent")

-- Create the store outside of the PurchasePromptApp so
-- we can utilize the store in APIs that can be accessed
-- by other modules
local function createStore()
	local abTest = ABTest.new()
	local network = Network.new()
	local analytics = Analytics.new()
	local platformInterface = PlatformInterface.new()
	local externalSettings = ExternalSettings.new()

	local publicBindables
	if FFlagHideAvatarIECPromptOnUpsellSuccess then
		publicBindables = PublicBindables.new({
			windowStateChangedBindable = windowStateChangedBindable
		})
	end

	store = Rodux.Store.new(Reducer, {}, {
		Thunk.middleware({
			[ABTest] = abTest,
			[Network] = network,
			[Analytics] = analytics,
			[PlatformInterface] = platformInterface,
			[ExternalSettings] = externalSettings,
			[PublicBindables] = if FFlagHideAvatarIECPromptOnUpsellSuccess then publicBindables else nil,
		}),
	})

	store.changed:connect(function(currentState, previousState)
		-- Check if the prompt has been opened or closed
		if previousState.windowState ~= currentState.windowState then
			windowStateChangedBindable:Fire({
				isShown = currentState.windowState == WindowState.Shown,
				hasCompletedPurchase = currentState.hasCompletedPurchase,
			})
		end

		-- Check if the prompt state has been set to PromptState.None
		if previousState.promptState ~= currentState.promptState and currentState.promptState == PromptState.None then
			promptStateSetToNoneBindable:Fire()
		end
	end)
end

local mountPurchasePrompt = function()
	if RunService:IsStudio() and RunService:IsEdit() or handle then
		return nil
	end

	createStore()
	local purchasePromptElement = Roact.createElement(PurchasePromptApp, {
		store = store
	})

	handle = Roact.mount(purchasePromptElement, CoreGui, "PurchasePromptApp")

	return handle
end

-- API for other modules to be able to initiate the purchase
-- of an avatar creation fee. Currently utilized by the
-- AvatarCreationService:PromptCreateAvatarAsync API flow
local function initiateAvatarCreationFeePurchase(avatarPublishMetadata, guid, humanoidModel, priceInRobux)
	if not store then
		error("initiateAvatarCreationFeePurchase cannot be called when the PurchasePrompt has not been mounted")
	end

	store:dispatch(initiateAvatarCreationFeePurchaseThunk(avatarPublishMetadata, guid, humanoidModel, priceInRobux))
end

return {
	mountPurchasePrompt = mountPurchasePrompt,
	initiateAvatarCreationFeePurchase = initiateAvatarCreationFeePurchase,
	-- This event fires when the window state is changed, i.e. prompt opens or closes.
	-- It returns isShown if the window is shown, and hasCompletedPurchase if the purchase was completed.
	windowStateChangedEvent = windowStateChangedBindable.Event,
	-- This event fires when the prompt state is set to PromptState.None
	promptStateSetToNoneEvent = promptStateSetToNoneBindable.Event,

	PublishAssetAnalytics = PublishAssetAnalytics,
}
