local Root = script:FindFirstAncestor("SceneUnderstanding")

local UserGameSettings = UserSettings():GetService("UserGameSettings")

local safelyAccessProperty = require(Root.safelyAccessProperty)
local getConnectedWires = require(Root.audio.getConnectedWires)

local AUDIO_GRAPH_CLASSES = {
	"Wire",
	"AudioAnalyzer",
	"AudioChorus",
	"AudioCompressor",
	"AudioDeviceInput",
	"AudioDeviceOutput",
	"AudioDistortion",
	"AudioEcho",
	"AudioEmitter",
	"AudioEqualizer",
	"AudioFader",
	"AudioFilter",
	"AudioFlanger",
	"AudioLimiter",
	"AudioListener",
	"AudioPitchShifter",
	"AudioPlayer",
	"AudioReverb",
}

type AudioGraphNode =
	Wire
	| AudioAnalyzer
	| AudioChorus
	| AudioCompressor
	| AudioDeviceInput
	| AudioDeviceOutput
	| AudioDistortion
	| AudioEcho
	| AudioEmitter
	| AudioEqualizer
	| AudioFader
	| AudioFilter
	| AudioFlanger
	| AudioLimiter
	| AudioListener
	| AudioPitchShifter
	| AudioPlayer
	| AudioReverb

local function toAudioGraphNode(instance: Instance?): AudioGraphNode?
	if instance then
		for _, class in AUDIO_GRAPH_CLASSES do
			if instance:IsA(class) then
				return (instance :: any) :: AudioGraphNode
			end
		end
	end
	return nil
end

local function calculateSoundPotentialAudibility(sound: Sound)
	if not sound.IsPlaying then
		return 0
	end

	-- These properties are integral to determining audibility but are not
	-- accessible in lower security levels. To ensure we don't accidentally
	-- lie about how audible something is we simply zero them out
	local rollOffGain = safelyAccessProperty(sound, "RollOffGain", 0)

	local groupVolume = if sound.SoundGroup then sound.SoundGroup.Volume else 1

	-- TODO MUS-1159: Add PlaybackLoudness as a factor for audibility. It looks
	-- to considerably increase accuracy
	return groupVolume * sound.Volume * rollOffGain
end

local function getOutputsOf(node: AudioGraphNode): { AudioGraphNode }
	if node:IsA("AudioEmitter") then
		return node:GetInteractingListeners() :: any
	end

	local outs: { AudioGraphNode } = {}
	for _, wire in getConnectedWires(node, "Output") do
		local target = toAudioGraphNode(wire.TargetInstance)
		if target then
			table.insert(outs, target)
		end
	end
	return outs
end

local function getAudibilityOf(node: AudioGraphNode, seen: { [AudioGraphNode]: boolean }?): number
	local seenHere = if seen then seen else {}
	if seenHere[node] then
		return 0
	end

	seenHere[node] = true
	local multiplier = 1
	if node:IsA("AudioPlayer") or node:IsA("AudioFader") then
		multiplier = node.Volume
	elseif node:IsA("AudioDeviceOutput") then
		return 1
	end

	local total = 0
	if node:IsA("AudioEmitter") then
		for _, listener in node:GetInteractingListeners() do
			-- TODO - MUS-1720: replace 1.0 with node:GetAudibilityFor(listener) once API is available
			total += 1.0 * getAudibilityOf(listener, seenHere)
		end
	else
		for _, output in getOutputsOf(node) do
			total += multiplier * getAudibilityOf(output, seenHere)
		end
	end

	return total
end

--[=[
	Determines the potential audibility of an audio source relative to the
	client.

	This function makes use of internal APIs and will return 0 in the following
	cases:
	1. [UserGameSettings.MasterVolume] is inaccessible
	2. For [Sound] instances, [Sound.RollOffGain] is inaccessible.

	@within SceneUnderstanding
	@tag internal
]=]
local function calculatePotentialAudibility(audioSource: AudioPlayer | Sound): number
	local masterVolume = safelyAccessProperty(UserGameSettings, "MasterVolume", 0)
	local potentialAudibility = 0

	if audioSource:IsA("Sound") then
		potentialAudibility = calculateSoundPotentialAudibility(audioSource)
	else
		potentialAudibility = getAudibilityOf(audioSource)
	end

	return masterVolume * potentialAudibility
end

return calculatePotentialAudibility
